<?php

declare(strict_types=1);

namespace Controllers;

use Models\User;
use Models\Exercise;
use Models\Comment;
use Models\WorkoutPlan;
use Models\ProgressLog;
use Models\Reminder;

class AdminController extends BaseController
{
    private function requireAdmin(): int
    {
        $userId = $this->requireAuth();
        $userModel = new User();
        if (!$userModel->isAdmin($userId)) {
            $this->redirect('/');
            exit;
        }
        return $userId;
    }

    public function dashboard(): void
    {
        $this->requireAdmin();
        $userModel = new User();
        $planModel = new WorkoutPlan();
        $commentModel = new Comment();

        $totalUsers = (int) $this->dbCount('users');
        $totalPlans = (int) $this->dbCount('workout_plans');
        $totalExercises = (int) $this->dbCount('exercises');
        $totalComments = $commentModel->countAll();
        $recentComments = $commentModel->getAllPaginated(10, 0);

        $this->adminView('admin/dashboard', [
            'title' => 'Admin Dashboard',
            'totalUsers' => $totalUsers,
            'totalPlans' => $totalPlans,
            'totalExercises' => $totalExercises,
            'totalComments' => $totalComments,
            'recentComments' => $recentComments,
        ]);
    }

    public function exercises(): void
    {
        $this->requireAdmin();
        $exerciseModel = new Exercise();
        $exercises = $exerciseModel->allWithDecoded();
        $this->adminView('admin/exercises/index', ['title' => 'Manage Exercises', 'exercises' => $exercises]);
    }

    public function exerciseCreate(): void
    {
        $this->requireAdmin();
        $this->adminView('admin/exercises/form', ['title' => 'Add Exercise', 'exercise' => null]);
    }

    public function exerciseEdit(): void
    {
        $this->requireAdmin();
        $id = (int) ($_GET['id'] ?? 0);
        $exerciseModel = new Exercise();
        $exercise = $id ? $exerciseModel->find($id) : null;
        if (!$exercise) {
            $this->redirect('/admin/exercises');
            return;
        }
        if (!empty($exercise['target_muscles'])) $exercise['target_muscles'] = json_decode($exercise['target_muscles'], true);
        if (!empty($exercise['equipment'])) $exercise['equipment'] = json_decode($exercise['equipment'], true);
        $this->adminView('admin/exercises/form', ['title' => 'Edit Exercise', 'exercise' => $exercise]);
    }

    public function exerciseSave(): void
    {
        $this->requireAdmin();
        $id = (int) ($_POST['id'] ?? 0);
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $difficulty = $_POST['difficulty'] ?? 'beginner';
        $targetMuscles = $_POST['target_muscles'] ?? [];
        $equipment = $_POST['equipment'] ?? [];
        if (is_string($targetMuscles)) $targetMuscles = array_filter(array_map('trim', explode(',', $targetMuscles)));
        if (is_string($equipment)) $equipment = array_filter(array_map('trim', explode(',', $equipment)));

        $exerciseModel = new Exercise();
        $data = [
            'name' => $name,
            'description' => $description,
            'difficulty' => $difficulty,
            'target_muscles' => json_encode(array_values($targetMuscles)),
            'equipment' => json_encode(array_values($equipment)),
            'demonstration_url' => trim($_POST['demonstration_url'] ?? ''),
        ];
        if ($id) {
            $exerciseModel->updateRecord($id, $data);
        } else {
            $exerciseModel->createRecord($data);
        }
        $this->redirect('/admin/exercises?updated=1');
    }

    public function exerciseDelete(): void
    {
        $this->requireAdmin();
        $id = (int) ($_POST['id'] ?? 0);
        if ($id) {
            $exerciseModel = new Exercise();
            $exerciseModel->delete($id);
        }
        $this->redirect('/admin/exercises');
    }

    public function users(): void
    {
        $this->requireAdmin();
        $userModel = new User();
        $users = $userModel->all('id', 'DESC');
        $this->adminView('admin/users', ['title' => 'Users', 'users' => $users]);
    }

    public function comments(): void
    {
        $this->requireAdmin();
        $commentModel = new Comment();
        $comments = $commentModel->getAllPaginated(100, 0);
        $this->adminView('admin/comments', ['title' => 'Comments', 'comments' => $comments]);
    }

    private function dbCount(string $table): int
    {
        $stmt = \Config\Database::getConnection()->query("SELECT COUNT(*) FROM {$table}");
        return (int) $stmt->fetchColumn();
    }

    protected function adminView(string $name, array $data = []): void
    {
        $data['_user_id'] = $_SESSION['user_id'] ?? null;
        $data['_user_email'] = $_SESSION['user_email'] ?? null;
        $data['base'] = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
        $data['_view_path'] = __DIR__ . '/../views/' . $name . '.php';
        extract($data);
        require __DIR__ . '/../views/admin/layout.php';
    }
}