<?php

declare(strict_types=1);

namespace Controllers;

use Models\WorkoutPlan;
use Models\PlanExercise;
use Models\Profile;
use Models\WorkoutGenerator;
use Models\ProgressLog;
use Models\WorkoutSession;
use Models\WorkoutSetLog;

class WorkoutPlanController extends BaseController
{
    public function index(): void
    {
        $userId = $this->requireAuth();
        $planModel = new WorkoutPlan();
        $plans = $planModel->getByUserId($userId);

        $this->view('plans/index', [
            'title' => 'My Workout Plans',
            'plans' => $plans,
        ]);
    }

    public function show(): void
    {
        $userId = $this->requireAuth();
        $planId = (int) ($_GET['id'] ?? 0);
        if (!$planId) {
            $this->redirect('/plans');
            return;
        }

        $planModel = new WorkoutPlan();
        $plan = $planModel->find($planId);
        if (!$plan || (int) $plan['user_id'] !== $userId) {
            $this->redirect('/plans');
            return;
        }

        $planExerciseModel = new PlanExercise();
        $byDay = $planExerciseModel->getByPlanIdGroupedByDay($planId);

        $this->view('plans/show', [
            'title' => 'Workout Plan',
            'plan' => $plan,
            'byDay' => $byDay,
        ]);
    }

    public function generate(): void
    {
        $userId = $this->requireAuth();
        $profileModel = new Profile();
        $planModel = new WorkoutPlan();
        $profile = $profileModel->getByUserId($userId);

        $active = $planModel->getActiveByUserId($userId);
        if ($active) {
            $planModel->setStatus((int) $active['id'], 'archived');
        }

        $generator = new WorkoutGenerator();
        $durationWeeks = (int) ($_POST['duration_weeks'] ?? 4);
        $planId = $generator->generateForUser($userId, $profile, $durationWeeks);

        $this->redirect('/plans?id=' . $planId);
    }

    public function setStatus(): void
    {
        $userId = $this->requireAuth();
        $planId = (int) ($_POST['plan_id'] ?? 0);
        $status = $_POST['status'] ?? '';
        if (!in_array($status, ['active', 'completed', 'archived'], true)) {
            $this->redirect('/plans');
            return;
        }

        $planModel = new WorkoutPlan();
        $plan = $planModel->find($planId);
        if (!$plan || (int) $plan['user_id'] !== $userId) {
            $this->redirect('/plans');
            return;
        }

        $planModel->setStatus($planId, $status);
        $this->redirect('/plans');
    }

    public function startWorkout(): void
    {
        $userId = $this->requireAuth();
        $planId = (int) ($_GET['plan_id'] ?? 0);
        $dayNumber = (int) ($_GET['day'] ?? 1);
        if (!$planId) {
            $this->redirect('/plans');
            return;
        }
        $planModel = new WorkoutPlan();
        $plan = $planModel->find($planId);
        if (!$plan || (int) $plan['user_id'] !== $userId) {
            $this->redirect('/plans');
            return;
        }
        $sessionModel = new WorkoutSession();
        $sessionDate = date('Y-m-d');
        $sessionId = $sessionModel->start($userId, $planId, $dayNumber, $sessionDate);
        $this->redirect($this->baseUrl() . '/plans/session?id=' . $sessionId);
    }

    public function session(): void
    {
        $userId = $this->requireAuth();
        $sessionId = (int) ($_GET['id'] ?? 0);
        if (!$sessionId) {
            $this->redirect('/plans');
            return;
        }
        $sessionModel = new WorkoutSession();
        $planExerciseModel = new PlanExercise();
        $session = $sessionModel->find($sessionId);
        if (!$session || (int) $session['user_id'] !== $userId) {
            $this->redirect('/plans');
            return;
        }
        $plan = (new WorkoutPlan())->find((int) $session['plan_id']);
        $exercises = $planExerciseModel->getByPlanId((int) $session['plan_id']);
        $dayExercises = array_filter($exercises, fn($e) => (int)$e['day_number'] === (int)$session['day_number']);
        $setLogModel = new WorkoutSetLog();
        $setLogs = $setLogModel->getBySessionId($sessionId);
        $this->view('plans/session', [
            'title' => 'Workout Session',
            'session' => $session,
            'plan' => $plan,
            'dayExercises' => $dayExercises,
            'setLogs' => $setLogs,
        ]);
    }

    public function logSet(): void
    {
        $userId = $this->requireAuth();
        $sessionId = (int) ($_POST['session_id'] ?? 0);
        $planExerciseId = (int) ($_POST['plan_exercise_id'] ?? 0);
        $setNumber = (int) ($_POST['set_number'] ?? 1);
        $reps = isset($_POST['reps']) ? (int) $_POST['reps'] : null;
        $weight = isset($_POST['weight']) ? (float) $_POST['weight'] : null;
        $sessionModel = new WorkoutSession();
        $session = $sessionModel->find($sessionId);
        if (!$session || (int) $session['user_id'] !== $userId) {
            $this->redirect('/plans');
            return;
        }
        $setLogModel = new WorkoutSetLog();
        $setLogModel->upsertForSession($sessionId, $planExerciseId, $setNumber, $reps, $weight, true, null);
        $this->redirect($this->baseUrl() . '/plans/session?id=' . $sessionId);
    }

    public function completeSession(): void
    {
        $userId = $this->requireAuth();
        $sessionId = (int) ($_POST['session_id'] ?? 0);
        $rating = isset($_POST['difficulty_rating']) ? (int) $_POST['difficulty_rating'] : null;
        $notes = trim($_POST['notes'] ?? '') ?: null;
        $sessionModel = new WorkoutSession();
        $sessionModel->complete($sessionId, $userId, $rating, $notes);
        $this->redirect('/plans?done=1');
    }

    private function baseUrl(): string
    {
        return rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
    }
}
