<?php

declare(strict_types=1);

namespace Models;

/**
 * Generates workout plans based on user profile (fitness level, goals, weekly time).
 */
class WorkoutGenerator
{
    private Exercise $exerciseModel;
    private WorkoutPlan $planModel;
    private PlanExercise $planExerciseModel;

    public function __construct()
    {
        $this->exerciseModel = new Exercise();
        $this->planModel = new WorkoutPlan();
        $this->planExerciseModel = new PlanExercise();
    }

    public function generateForUser(int $userId, ?array $profile, int $durationWeeks = 4): int
    {
        $exercises = $this->exerciseModel->allWithDecoded();
        $fitnessLevel = $profile['fitness_level'] ?? 3;
        $weeklyMinutes = (int) ($profile['weekly_time_available'] ?? 180);
        $goals = $profile['goals'] ?? [];
        if (!is_array($goals)) $goals = [];

        $difficulty = $this->mapFitnessToDifficulty($fitnessLevel);
        $filtered = array_filter($exercises, fn($e) => $e['difficulty'] === $difficulty || $this->isEasier($e['difficulty'], $difficulty));

        if (empty($filtered)) {
            $filtered = $exercises;
        }

        $planId = $this->planModel->create($userId, $durationWeeks);
        $daysPerWeek = min(5, max(2, (int) floor($weeklyMinutes / 45)));
        $exercisesPerDay = min(5, max(3, (int) ceil(count($filtered) / 7)));
        $pool = array_values($filtered);
        $used = [];

        $dayNumbers = [];
        for ($day = 1; $day <= $daysPerWeek; $day++) {
            $dayNumbers[] = $day;
            $picked = $this->pickExercises($pool, $exercisesPerDay, $used);
            foreach ($picked as $ex) {
                $sets = $fitnessLevel >= 4 ? 4 : 3;
                $reps = $fitnessLevel <= 2 ? '12-15' : ($fitnessLevel <= 4 ? '10-12' : '8-10');
                $this->planExerciseModel->addToPlan(
                    $planId,
                    (int) $ex['id'],
                    $day,
                    $sets,
                    $reps,
                    'moderate',
                    60
                );
            }
        }

        // Auto-create reminders for each workout day
        try {
            $plan = $this->planModel->find($planId);
            if ($plan && !empty($dayNumbers)) {
                $reminderService = new ReminderService();
                $reminderService->createRemindersForPlan(
                    (int) $plan['user_id'],
                    $planId,
                    $plan['start_date'],
                    (int) $plan['duration_weeks'],
                    $dayNumbers
                );
            }
        } catch (\Throwable $e) {
            // Don't fail plan creation if reminders table doesn't exist yet
        }

        return $planId;
    }

    private function mapFitnessToDifficulty(int $level): string
    {
        if ($level <= 2) return 'beginner';
        if ($level <= 4) return 'intermediate';
        return 'advanced';
    }

    private function isEasier(string $exDiff, string $target): bool
    {
        $order = ['beginner' => 1, 'intermediate' => 2, 'advanced' => 3];
        return ($order[$exDiff] ?? 0) <= ($order[$target] ?? 3);
    }

    private function pickExercises(array $pool, int $count, array &$used): array
    {
        $available = array_filter($pool, fn($e) => !in_array((int)$e['id'], $used, true));
        if (count($available) < $count) {
            $available = $pool;
            $used = [];
        }
        shuffle($available);
        $picked = array_slice($available, 0, $count);
        foreach ($picked as $e) {
            $used[] = (int) $e['id'];
        }
        return $picked;
    }
}
